import discord
from discord.ext import commands
from discord import app_commands, ui
import os
import json

TOKEN = "YOUR_BOT_TOKEN"

intents = discord.Intents.default()
intents.message_content = True
intents.guilds = True
intents.members = True

bot = commands.Bot(command_prefix='cl$', intents=intents)
bot.remove_command("help")

#Colors
purple= 0x4f00ff
red= 0xff0000
blue= 0x0015ff
green= 0x00ff00

# --- Utility functions ---
def load_config():
    if os.path.exists("chat_logs.json"):
        try:
            with open("chat_logs.json", "r") as f:
                return json.load(f)
        except json.JSONDecodeError:
            print("⚠️ Corrupted config file. Resetting.")
            with open("chat_logs.json", "w") as f:
                json.dump({}, f, indent=4)
            return {}
    else:
        with open("chat_logs.json", "w") as f:
            json.dump({}, f, indent=4)
        return {}

def save_config(config):
    with open("chat_logs.json", "w") as f:
        json.dump(config, f, indent=4)

def get_text_channels(guild):
    return [ch for ch in guild.text_channels if ch.permissions_for(guild.me).read_messages]

def get_members(guild):
    return [m for m in guild.members if not m.bot]

def get_roles(guild):
    return [r for r in guild.roles if not r.is_default()]

# --- Views ---
class LogChannelSelect(ui.Select):
    def __init__(self, guild):
        options = [
            discord.SelectOption(label=ch.name, value=str(ch.id)) for ch in get_text_channels(guild)
        ]
        super().__init__(placeholder="Select a log channel", min_values=1, max_values=1, options=options)

    async def callback(self, interaction: discord.Interaction):
        config = load_config()
        guild_id = str(interaction.guild.id)
        config[guild_id] = {
            "log_channel": int(self.values[0]),
            "filter_channels": [],
            "filter_users": []
        }
        save_config(config)
        embed = discord.Embed(
            title="✅ Logging Enabled",
            description="Logging has been enabled in the selected channel.",
            color=green
        )
        await interaction.response.send_message(embed=embed, ephemeral=True)

class EnableLoggingView(ui.View):
    def __init__(self, interaction):
        super().__init__(timeout=60)
        self.add_item(LogChannelSelect(interaction.guild))

class ChannelFilterSelect(ui.Select):
    def __init__(self, guild):
        options = [
            discord.SelectOption(label=ch.name, value=str(ch.id)) for ch in get_text_channels(guild)
        ]
        super().__init__(placeholder="Select channels to log", min_values=0, max_values=len(options), options=options)

    async def callback(self, interaction: discord.Interaction):
        config = load_config()
        guild_id = str(interaction.guild.id)
        if guild_id not in config:
            embed = discord.Embed(
                title="⚠️ Logging Not Enabled",
                description="Please enable logging first using the panel.",
                color=red
            )
            await interaction.response.send_message(embed=embed, ephemeral=True)
            return
        config[guild_id]["filter_channels"] = [int(v) for v in self.values]
        save_config(config)
        embed = discord.Embed(
            title="✅ Channel Filters Updated",
            description="Selected channels will now be logged.",
            color=green
        )
        await interaction.response.send_message(embed=embed, ephemeral=True)

class UserFilterSelect(ui.Select):
    def __init__(self, guild):
        options = [
            discord.SelectOption(label=m.display_name, value=str(m.id)) for m in get_members(guild)[:25]
        ]
        super().__init__(placeholder="Select users to log", min_values=0, max_values=len(options), options=options)

    async def callback(self, interaction: discord.Interaction):
        config = load_config()
        guild_id = str(interaction.guild.id)
        if guild_id not in config:
            embed = discord.Embed(
                title="⚠️ Logging Not Enabled",
                description="Please enable logging first using the panel.",
                color=red
            )
            await interaction.response.send_message(embed=embed, ephemeral=True)
            return
        config[guild_id]["filter_users"] = [int(v) for v in self.values]
        save_config(config)
        embed = discord.Embed(
            title="✅ User Filters Updated",
            description="Selected users' messages will now be logged.",
            color=green
        )
        await interaction.response.send_message(embed=embed, ephemeral=True)

class RoleFilterSelect(ui.Select):
    def __init__(self, guild):
        options = [
            discord.SelectOption(label=r.name, value=str(r.id)) for r in get_roles(guild)
        ]
        super().__init__(placeholder="Select roles to log", min_values=0, max_values=len(options), options=options)

    async def callback(self, interaction: discord.Interaction):
        config = load_config()
        guild_id = str(interaction.guild.id)
        if guild_id not in config:
            embed = discord.Embed(
                title="⚠️ Logging Not Enabled",
                description="Please enable logging first using the panel.",
                color=red
            )
            await interaction.response.send_message(embed=embed, ephemeral=True)
            return
        config[guild_id]["filter_roles"] = [int(v) for v in self.values]
        save_config(config)
        embed = discord.Embed(
            title="✅ Role Filters Updated",
            description="Selected roles' messages will now be logged.",
            color=green
        )
        await interaction.response.send_message(embed=embed, ephemeral=True)

class FilterEditView(ui.View):
    def __init__(self, interaction):
        super().__init__(timeout=60)
        self.add_item(ChannelFilterSelect(interaction.guild))
        self.add_item(UserFilterSelect(interaction.guild))
        self.add_item(RoleFilterSelect(interaction.guild))

class ChatLogView(ui.View):
    def __init__(self, interaction: discord.Interaction):
        super().__init__(timeout=60)
        self.interaction = interaction

    @ui.button(label="Enable Logging", style=discord.ButtonStyle.success)
    async def enable(self, interaction: discord.Interaction, button: ui.Button):
        embed = discord.Embed(
            title="📥 Enable Logging",
            description="Select a log channel below to begin logging messages.",
            color=green
        )
        await interaction.response.send_message(embed=embed, view=EnableLoggingView(interaction), ephemeral=True)

    @ui.button(label="View Settings", style=discord.ButtonStyle.blurple)
    async def view_settings(self, interaction: discord.Interaction, button: ui.Button):
        config = load_config()
        guild_id = str(interaction.guild.id)
        settings = config.get(guild_id)
        if not settings:
            embed = discord.Embed(
                title="⚠️ Not Configured",
                description="Chat logging is not set up yet.",
                color=red
            )
            await interaction.response.send_message(embed=embed, ephemeral=True)
            return
        channel = interaction.guild.get_channel(settings["log_channel"])
        channels = [interaction.guild.get_channel(cid) for cid in settings["filter_channels"]]
        users = [interaction.guild.get_member(uid) for uid in settings["filter_users"]]
        desc = (
            f"**Log Channel:** {channel.mention if channel else 'Not found'}\n"
            f"**Filtered Channels:** {', '.join(c.mention for c in channels if c) or 'All'}\n"
            f"**Filtered Users:** {', '.join(u.mention for u in users if u) or 'All'}"
        )
        embed = discord.Embed(
            title="📄 Current Logging Settings",
            description=desc,
            color=purple
        )
        await interaction.response.send_message(embed=embed, ephemeral=True)

    @ui.button(label="Edit Filters", style=discord.ButtonStyle.gray)
    async def edit_filters(self, interaction: discord.Interaction, button: ui.Button):
        embed = discord.Embed(
            title="🎛️ Edit Logging Filters",
            description="Select channels and users to filter log entries.",
            color=purple
        )
        await interaction.response.send_message(embed=embed, view=FilterEditView(interaction), ephemeral=True)

    @ui.button(label="Remove Logging", style=discord.ButtonStyle.danger)
    async def remove(self, interaction: discord.Interaction, button: ui.Button):
        config = load_config()
        guild_id = str(interaction.guild.id)
        if guild_id in config:
            del config[guild_id]
            save_config(config)
            embed = discord.Embed(
                title="🗑️ Logging Disabled",
                description="Chat logging has been disabled for this server.",
                color=red
            )
        else:
            embed = discord.Embed(
                title="⚠️ Not Enabled",
                description="Logging was not enabled in this server.",
                color=red
            )
        await interaction.response.send_message(embed=embed, ephemeral=True)

@bot.tree.command(name="panel", description="Admin panel to manage chat log settings")
@app_commands.checks.has_permissions(administrator=True)
async def panel(interaction: discord.Interaction):
    embed = discord.Embed(
        title="🛠️ Chat Log Admin Panel",
        description="Use the buttons below to manage logging.",
        color=purple
    )
    await interaction.response.send_message(embed=embed, view=ChatLogView(interaction), ephemeral=True)

@bot.tree.command(name="add-filterd-words", description="Filter messages containing specific words")
@app_commands.describe(words="Comma-separated list of words to filter")
@app_commands.checks.has_permissions(administrator=True)
async def add_filter_words(interaction: discord.Interaction, words: str):
    config = load_config()
    guild_id = str(interaction.guild.id)
    if guild_id not in config:
        config[guild_id] = {}
    if "filter_words" not in config[guild_id]:
        config[guild_id]["filter_words"] = []
    new_words = [word.strip() for word in words.split(",") if word.strip()]
    config[guild_id]["filter_words"].extend(new_words)
    save_config(config)
    embed = discord.Embed(
        title="✅ Filter Words Added",
        description=f"Added words: {', '.join(new_words)}",
        color=green
    )
    await interaction.response.send_message(embed=embed, ephemeral=True)

@bot.tree.command(name="remove-filterd-words", description="Remove filtered words")
@app_commands.describe(words="Comma-separated list of words to remove")
@app_commands.checks.has_permissions(administrator=True)
async def remove_filter_words(interaction: discord.Interaction, words: str):
    config = load_config()
    guild_id = str(interaction.guild.id)
    if guild_id not in config or "filter_words" not in config[guild_id]:
        embed = discord.Embed(
            title="⚠️ No Filter Words Set",
            description="No filter words have been set for this server.",
            color=red
        )
        await interaction.response.send_message(embed=embed, ephemeral=True)
        return
    current_words = config[guild_id]["filter_words"]
    words_to_remove = [word.strip() for word in words.split(",") if word.strip()]
    config[guild_id]["filter_words"] = [word for word in current_words if word not in words_to_remove]
    save_config(config)
    embed = discord.Embed(
        title="✅ Filter Words Removed",
        description=f"Removed words: {', '.join(words_to_remove)}",
        color=green
    )
    await interaction.response.send_message(embed=embed, ephemeral=True)

@bot.tree.command(name="list-filterd-words", description="List all filtered words")
@app_commands.checks.has_permissions(administrator=True)
async def list_filter_words(interaction: discord.Interaction):
    config = load_config()
    guild_id = str(interaction.guild.id)
    if guild_id not in config or "filter_words" not in config[guild_id]:
        embed = discord.Embed(
            title="⚠️ No Filter Words Set",
            description="No filter words have been set for this server.",
            color=red
        )
        await interaction.response.send_message(embed=embed, ephemeral=True)
        return
    words = config[guild_id]["filter_words"]
    embed = discord.Embed(
        title="📜 Filtered Words",
        description=", ".join(words) if words else "No words filtered.",
        color=purple
    )
    await interaction.response.send_message(embed=embed, ephemeral=True)

@bot.tree.command(name="clear", description="Clear channel")
async def clear(interaction: discord.Interaction, amount: int = 100):
    # ignore if not in guild
    if not interaction.guild:
        embed = discord.Embed(
            title="⚠️ Not in a Guild",
            description="This command can only be used in a server.",
            color=red
        )
        await interaction.response.send_message(embed=embed, ephemeral=True)
        return
    manage_messages = discord.Interaction.user.guild_permissions.manage_messages
    if manage_messages == True:
        if amount < 1 or amount > 1000:
            embed = discord.Embed(
                title="⚠️ Invalid Amount",
                description="Please specify a number between 1 and 1000.",
                color=red
            )
            await interaction.response.send_message(embed=embed, ephemeral=True)
            return
        await interaction.response.defer(ephemeral=True)
        deleted = await interaction.channel.purge(limit=amount)
        embed = discord.Embed(
            title="🧹 Messages Cleared",
            description=f"Successfully cleared {len(deleted)} messages.",
            color=green
        )
        await interaction.followup.send(embed=embed, ephemeral=True)
    else:
        embed = discord.Embed(
            title="⚠️ Missing Permissions",
            description="You need the `Manage Messages` permission to use this command.",
            color=red
        )
        await interaction.response.send_message(embed=embed, ephemeral=True)
        return

# --- Message Logger ---
@bot.event
async def on_message(message):
    # Prevent duplicate processing and ignore bots/DMs
    await bot.process_commands(message)
    if message.author.bot or not message.guild:
        return

    config = load_config()
    guild_id = str(message.guild.id)
    if guild_id not in config:
        return
    settings = config[guild_id]
    if "filter_words" in settings:
        for word in settings["filter_words"]:
            if word.lower() in message.content.lower():
                await message.delete()
                embed = discord.Embed(
                    title="🗑️ Message Deleted",
                    description=f"Your message was deleted for containing following word",
                    color=red
                )
                embed.add_field(name="Filtered Word", value=word, inline=False)
                embed.add_field(name="Server", value=message.guild.name, inline=False)
                await message.author.send(embed=embed)
                break

    if message.author.bot or not message.guild:
        return
    config = load_config()
    guild_id = str(message.guild.id)
    if guild_id not in config:
        return
    settings = config[guild_id]
    log_channel = message.guild.get_channel(settings["log_channel"])
    if not log_channel:
        return
    if settings["filter_channels"] and message.channel.id not in settings["filter_channels"]:
        return
    if settings["filter_users"] and message.author.id not in settings["filter_users"]:
        return
    if settings.get("filter_roles"):
        author_role_ids = [role.id for role in message.author.roles]
        if not any(role_id in author_role_ids for role_id in settings["filter_roles"]):
            return
    if log_channel.id == message.channel.id:
        return
    embed = discord.Embed(
        title="📝 Message Logged",
        url=message.jump_url,
        color=purple
    )
    embed.add_field(name="Channel", value=message.channel.mention, inline=True)
    embed.add_field(name="Author", value=message.author.mention, inline=True)
    embed.add_field(name="Message Content", value=message.content or "No content", inline=False)
    embed.add_field(name="Message ID", value=message.id, inline=True)
    embed.add_field(name="Author ID", value=message.author.id, inline=True)
    embed.set_footer(text=f"Made by PyZen | discord.gg/m6FUx2sXXc")
    if message.attachments:
        embed.add_field(name="Attachments", value=", ".join([f"[{a.filename}]({a.url})" for a in message.attachments]), inline=False)
    embed.timestamp = message.created_at
    await log_channel.send(embed=embed)

@bot.event
async def on_message_edit(before, after):
    if before.author.bot or not before.guild:
        return
    config = load_config()
    guild_id = str(before.guild.id)
    if guild_id not in config:
        return
    settings = config[guild_id]
    log_channel = before.guild.get_channel(settings["log_channel"])
    if not log_channel:
        return
    if settings["filter_channels"] and before.channel.id not in settings["filter_channels"]:
        return
    if settings["filter_users"] and before.author.id not in settings["filter_users"]:
        return
    if settings.get("filter_roles"):
        author_role_ids = [role.id for role in before.author.roles]
        if not any(role_id in author_role_ids for role_id in settings["filter_roles"]):
            return
    if log_channel.id == before.channel.id:
        return
    embed = discord.Embed(
        title="📝 Message Edited",
        url=before.jump_url,
        color=blue
    )
    embed.add_field(name="Channel", value=before.channel.mention, inline=True)
    embed.add_field(name="Author", value=before.author.mention, inline=True)
    embed.add_field(name="Before", value=before.content or "No content", inline=False)
    embed.add_field(name="After", value=after.content or "No content", inline=False)
    embed.add_field(name="Message ID", value=before.id, inline=True)
    embed.add_field(name="Author ID", value=before.author.id, inline=True)
    embed.set_footer(text=f"Made by PyZen | discord.gg/m6FUx2sXXc")
    embed.timestamp = after.edited_at or after.created_at
    await log_channel.send(embed=embed)

@bot.event
async def on_message_delete(message):
    if message.author.bot or not message.guild:
        return
    config = load_config()
    guild_id = str(message.guild.id)
    if guild_id not in config:
        return
    settings = config[guild_id]
    log_channel = message.guild.get_channel(settings["log_channel"])
    if not log_channel:
        return
    if settings["filter_channels"] and message.channel.id not in settings["filter_channels"]:
        return
    if settings["filter_users"] and message.author.id not in settings["filter_users"]:
        return
    if settings.get("filter_roles"):
        author_role_ids = [role.id for role in message.author.roles]
        if not any(role_id in author_role_ids for role_id in settings["filter_roles"]):
            return
    if log_channel.id == message.channel.id:
        return
    embed = discord.Embed(
        title="🗑️ Message Deleted",
        url=message.jump_url,
        color=red
    )
    embed.add_field(name="Channel", value=message.channel.mention, inline=True)
    embed.add_field(name="Author", value=message.author.mention, inline=True)
    embed.add_field(name="Message Content", value=message.content or "No content", inline=False)
    embed.add_field(name="Message ID", value=message.id, inline=True)
    embed.add_field(name="Author ID", value=message.author.id, inline=True)
    embed.set_footer(text=f"Made by PyZen | discord.gg/m6FUx2sXXc")
    embed.timestamp = message.created_at
    await log_channel.send(embed=embed)

# --- Startup ---
@bot.event
async def on_ready():
    print("-------------------------------------------------------------")
    print(f"Logged in as {bot.user}")
    print(f"User ID: {bot.user.id}")
    print("-------------------------------------------------------------")
    try:
        await bot.change_presence(activity=discord.CustomActivity(name="Made by PyZen | discord.gg/m6FUx2sXXc"))
        synced = await bot.tree.sync()
        print(f"Synced {len(synced)} command(s).")
        for guild in bot.guilds:
            print("-------------------------------------------------------------")
            print(f"Connected to guild: {guild.name} (ID: {guild.id})")
    except Exception as e:
        print(f"Failed to sync commands: {e}")


bot.run(TOKEN)
